const CardRequest = require('../models/Card');
const User = require('../models/Users');
const { cardApprovedEmail, cardDeclinedEmail } = require('../utils/emailTemplates');
const sendEmail = require('../utils/sendEmail');

exports.getAllCardRequests = async (req, res) => {
  try {
    const cards = await CardRequest.find().populate('userId', 'name email').sort({ createdAt: -1 });
    res.json(cards);
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};
exports.getSingleCardRequest = async (req, res) => {
  try {
    const card = await CardRequest.findById(req.params.id).populate('userId', 'name email');
    if (!card) return res.status(404).json({ message: 'Card request not found' });
    res.json(card);
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};
exports.approveCard = async (req, res) => {
  try {
    const card = await CardRequest.findById(req.params.id).populate('userId');
    if (!card) return res.status(404).json({ message: 'Card request not found' });

    card.status = 'approved';
    await card.save();

    // Send approval email
    await sendEmail(
      card.userId.email,
      'Card Request Approved',
      `Your ${card.cardType} card request has been approved.`,
      cardApprovedEmail(card.userId.name, card.cardType)
    );

    res.json({ message: 'Card request approved and email sent', card });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

exports.declineCard = async (req, res) => {
  try {
    const card = await CardRequest.findById(req.params.id).populate('userId');
    if (!card) return res.status(404).json({ message: 'Card request not found' });

    card.status = 'declined';

    const user = await User.findById(card.userId._id);
    if (user) {
      user.balance += 2000;
      await user.save();

      // Send decline email
      await sendEmail(
        user.email,
        'Card Request Declined',
        `Your ${card.cardType} card request was declined.`,
        cardDeclinedEmail(user.name, card.cardType)
      );
    }

    await card.save();
    res.json({ message: 'Card request declined, user notified and refunded', card });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

