const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const User = require('../models/Users'); // Ensure this matches your filename
const { generateVerificationCode } = require('../utils/generateCode');
const sendEmail = require('../utils/sendEmail');
const { getVerificationEmailTemplate } = require('../utils/emailTemplates');
const Deposit = require('../models/Deposit');
const Withdrawal = require('../models/Withdrawal');
const SupportMessage = require("../models/SupportMessage");
const cloudinary = require('../utils/cloudinary');


const generateUniqueAccountNumber = async () => {
  const prefix = '01'; 
  let accountNumber;
  let exists = true;

  while (exists) {
    const randomPart = Math.floor(100000000 + Math.random() * 900000000).toString(); // 8 digits
    accountNumber = prefix + randomPart; // Total = 10 digits
    exists = await User.exists({ accountNumber });
  }

  return accountNumber;
};

// --- REGISTER (Updated for 4-Step Wizard) ---
exports.register = async (req, res) => {
  try {
    const {
      // Step 1: Personal
      firstName, lastName, middleName, username,
      // Step 2: Contact
      email, phone, country,
      // Step 3: Account Preferences
      currency, accountType, transactionPin,
      // Step 4: Security
      password, agreeTerms
    } = req.body;

    // 1. Check for existing user (Email or Username)
    let existingUser = await User.findOne({ $or: [{ email }, { username }] });
    if (existingUser) {
      if (existingUser.email === email) return res.status(400).json({ message: 'Email already exists' });
      if (existingUser.username === username) return res.status(400).json({ message: 'Username already taken' });
    }

    // 2. Hash Password and PIN
    const hashedPassword = await bcrypt.hash(password, 10);
    
    // Ensure PIN is 4 digits before hashing
    if (!/^\d{4}$/.test(transactionPin)) {
      return res.status(400).json({ message: 'PIN must be exactly 4 digits' });
    }
    // const hashedPin = await bcrypt.hash(transactionPin, 10);

    // 3. Generate Account Number
    const accountNumber = await generateUniqueAccountNumber();
    if (!accountNumber) {
      return res.status(500).json({ message: 'Error generating account number' });
    }

    // 4. Create User
    const user = new User({
      firstName,
      lastName,
      middleName,
      username,
      email,
      phoneNumber:phone,
      country,
      currency,
      accountType,
      pin: transactionPin,
      password: hashedPassword,
      termsAgreed:agreeTerms,
      accountNumber,
      pinSet: true, 
      detailsSet: true
    });

    await user.save();

    // Optional: Send Welcome Email here
    // await sendEmail(user.email, 'Welcome', 'Welcome to our banking portal', ...);

    const token = jwt.sign(
      { id: user._id, email: user.email, isAdmin: user.isAdmin }, 
      process.env.JWT_SECRET, 
      { expiresIn: '1d' }
    );

    res.status(201).json({ 
      message: 'Account created successfully', 
      token,
      accountNumber,
      user: {
        id: user._id,
        email: user.email,
        username: user.username
      }
    });

  } catch (err) {
    console.error("Registration Error:", err);
    res.status(500).json({ message: 'Server error', error: err.message });
  }
};

// --- LOGIN ---
exports.login = async (req, res) => {
  const { email, password } = req.body;

  try {
    // Find user by email
    const user = await User.findOne({ email });
    if (!user) return res.status(400).json({ message: 'Invalid credentials' });

    // Check Password
    const isMatch = await bcrypt.compare(password, user.password);
    if (!isMatch) return res.status(400).json({ message: 'Invalid credentials' });

    // Update last login
    user.lastLogin = new Date();
    await user.save();

    // Create Token
    const token = jwt.sign(
      { id: user._id, email: user.email, isAdmin: user.isAdmin }, 
      process.env.JWT_SECRET, 
      { expiresIn: '1d' }
    );

    res.json({
      token,
      user: {
        _id: user._id,
        firstName: user.firstName,
        lastName: user.lastName,
        username: user.username,
        email: user.email,
        accountNumber: user.accountNumber,
        emailVerified: user.emailVerified,
        pinSet: user.pinSet,
        isAdmin: user.isAdmin,
        profilePicture: user.profilePicture
      }
    });

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- GET CURRENT USER ---
exports.getMe = async (req, res) => {
  try {
    const user = await User.findById(req.user.id).select('-password -pin');
    res.json(user);
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};

exports.updatePin = async (req, res) => {
  const { currentPin, newPin } = req.body;

  if (!/^\d{4}$/.test(newPin)) {
    return res.status(400).json({ message: 'New PIN must be exactly 4 digits' });
  }

  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    
    if (currentPin != user.pin) return res.status(400).json({ message: 'Current PIN is incorrect' });

    user.pin = newPin
    await user.save();

    res.json({ message: 'PIN updated successfully' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};
exports.enableTwoStep = async (req, res) => {
  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    user.twoStepEnabled = true;
    await user.save();

    res.json({ message: 'Two-step verification enabled' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
}
exports.changePassword = async (req, res) => {
  const { currentPassword, newPassword } = req.body;

  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    const isMatch = await bcrypt.compare(currentPassword, user.password);
    if (!isMatch) return res.status(400).json({ message: 'Current password is incorrect' });

    user.password = await bcrypt.hash(newPassword, 10);
    await user.save();

    res.json({ message: 'Password updated successfully' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- COMPLETE PROFILE (Modified for KYC / Identity only now) ---
// Since basic info is done in Register, this is now for ID Verification
exports.completeProfile = async (req, res) => {
  const { idType, idNumber, gender, dob } = req.body; // These weren't in screenshots, assuming they are "Step 5" or KYC

  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    // Update KYC fields
    if(idType) user.idType = idType;
    if(idNumber) user.idNumber = idNumber;
    if(gender) user.gender = gender;
    if(dob) user.dob = dob;
    
    // Generate Email Verification Code (if not done yet)
    if (!user.emailVerified) {
        const code = generateVerificationCode();
        user.verificationCode = code;
        user.verificationCodeExpires = new Date(Date.now() + 15 * 60 * 1000); // 15 min
        
        const htmlTemplate = getVerificationEmailTemplate(user.firstName, code);
        await sendEmail(user.email, 'Email Verification Code', `Your code is: ${code}`, htmlTemplate);
    }

    await user.save();
    res.json({ message: 'KYC Information updated. Please check email for verification code.' });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- VERIFY EMAIL ---
exports.verifyEmailCode = async (req, res) => {
  const { code } = req.body;

  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    if (
      user.verificationCode !== code ||
      new Date(user.verificationCodeExpires) < new Date()
    ) {
      return res.status(400).json({ message: 'Invalid or expired code' });
    }

    user.emailVerified = true;
    user.verificationCode = null;
    user.verificationCodeExpires = null;
    await user.save();

    res.json({ message: 'Email verified successfully.' });
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};

// --- SET PIN (Optional - if they want to change it later) ---
exports.setPin = async (req, res) => {
  const { pin } = req.body;

  if (!/^\d{4}$/.test(pin)) {
    return res.status(400).json({ message: 'PIN must be 4 digits.' });
  }

  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    user.pin = await bcrypt.hash(pin, 10);
    user.pinSet = true;
    await user.save();

    res.json({ message: 'Transaction PIN updated successfully.' });
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};

// --- TRANSACTIONS & OTHERS ---
exports.getUserTransactions = async (req, res) => {
  try {
    const [deposits, withdrawals] = await Promise.all([
      Deposit.find({ userId: req.user.id }),
      Withdrawal.find({ userId: req.user.id }),
    ]);

    const transactions = [
      ...deposits.map(tx => ({
        _id: tx._id,
        type: 'deposit',
        amount: tx.amount,
        status: tx.status,
        createdAt: tx.createdAt,
      })),
      ...withdrawals.map(tx => ({
        _id: tx._id,
        type: 'withdrawal',
        amount: tx.amount,
        status: tx.status,
        createdAt: tx.createdAt,
      }))
    ];

    transactions.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

    res.json({ transactions });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- SUPPORT MESSAGES ---
exports.sendMessage = async (req, res) => {
  try {
    const { message } = req.body;
    if (!message) return res.status(400).json({ message: "Message is required" });

    const supportMessage = new SupportMessage({
      userId: req.user.id,
      message,
    });

    await supportMessage.save();
    res.status(201).json({ message: "Message sent successfully", supportMessage });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: "Server error" });
  }
};

exports.getMyMessages = async (req, res) => {
  try {
    const messages = await SupportMessage.find({ userId: req.user.id }).sort({ createdAt: -1 });
    res.json({ messages });
  } catch (err) {
    res.status(500).json({ message: "Server error" });
  }
};

exports.getAllMessages = async (req, res) => {
  try {
    const messages = await SupportMessage.find().populate("userId", "email firstName lastName").sort({ createdAt: -1 });
    res.json({ messages });
  } catch (err) {
    res.status(500).json({ message: "Server error" });
  }
};

exports.replyMessage = async (req, res) => {
  try {
    const { messageId } = req.params;
    const { reply } = req.body;

    const message = await SupportMessage.findById(messageId);
    if (!message) return res.status(404).json({ message: "Message not found" });

    message.reply = reply;
    await message.save();

    res.json({ message: "Reply sent", updated: message });
  } catch (err) {
    res.status(500).json({ message: "Server error" });
  }
};

// --- PROFILE PICTURE ---
exports.uploadProfilePicture = async (req, res) => {
  try {
    if (!req.file) return res.status(400).json({ message: 'No image uploaded' });

    const result = await cloudinary.uploader.upload_stream(
      {
        folder: 'profile_pictures',
        resource_type: 'image',
        public_id: `user_${req.user.id}_${Date.now()}`,
      },
      async (error, result) => {
        if (error) {
          console.error(error);
          return res.status(500).json({ message: 'Cloudinary upload failed' });
        }
        
        const user = await User.findByIdAndUpdate(
          req.user.id,
          { profilePicture: result.secure_url },
          { new: true }
        );
        
        res.json({
          message: 'Profile picture updated',
          profilePicture: user.profilePicture,
        });
      }
    );

    result.end(req.file.buffer);

  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- ADMIN: USER MANAGEMENT ---
exports.getAllUsers = async (req, res) => {
  try {
    const users = await User.find().select('-password -pin');
    res.json({ users });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
}

exports.getUserById = async (req, res) => {
  const { id } = req.params;
  try {
    const user = await User.findById(id).select('-password -verificationCode -verificationCodeExpires -pin -__v')
    if (!user) return res.status(404).json({ message: 'User not found' });
    res.json(user);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- ADMIN: WITHDRAWALS ---
exports.getAllWithdrawals = async (req, res) => {
  try {
    const withdrawals = await Withdrawal.find().populate('userId', 'firstName lastName email accountNumber').sort({ createdAt: -1 });
    res.json({ withdrawals });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
}

exports.getSingleWithdrawal = async (req, res) => {
  const { id } = req.params;
  try {
    const withdrawal = await Withdrawal.findById(id).populate('userId', 'firstName lastName email accountNumber');
    if (!withdrawal) return res.status(404).json({ message: 'Withdrawal not found' });
    res.json(withdrawal);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
}

exports.approveOrRejectWithdrawal = async (req, res) => {
  const { id } = req.params;
  const { status } = req.body; 

  if (!['approved', 'rejected'].includes(status)) {
    return res.status(400).json({ message: 'Invalid status' });
  }

  try {
    const withdrawal = await Withdrawal.findById(id);
    if (!withdrawal) return res.status(404).json({ message: 'Withdrawal not found' });

    if (withdrawal.status === 'approved' || withdrawal.status === 'rejected') {
      return res.status(400).json({ message: `Withdrawal already ${withdrawal.status}` });
    }

    withdrawal.status = status;
    await withdrawal.save();

    if (status === 'approved') {
      const user = await User.findById(withdrawal.userId);
      if (!user) return res.status(404).json({ message: 'User not found' });

      if (user.balance < withdrawal.amount) {
        return res.status(400).json({ message: 'Insufficient balance' });
      }

      user.balance -= withdrawal.amount;
      await user.save();
    }

    res.json({ message: `Withdrawal ${status}`, withdrawal });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- ADMIN: DEPOSITS ---
exports.getAllDeposits = async (req, res) => {
  try {
    const deposits = await Deposit.find().populate('userId', 'firstName lastName email accountNumber').sort({ createdAt: -1 });
    res.json({ deposits });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
}

exports.getSingleDeposit = async (req, res) => {
  const { id } = req.params;
  try {
    const deposit = await Deposit.findById(id).populate('userId', 'firstName lastName email accountNumber');
    if (!deposit) return res.status(404).json({ message: 'Deposit not found' });
    res.json(deposit);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
}

exports.approveOrRejectDeposit = async (req, res) => {
  const { id } = req.params;
  const { status } = req.body; 

  if (!['approved', 'rejected'].includes(status)) {
    return res.status(400).json({ message: 'Invalid status' });
  }

  try {
    const deposit = await Deposit.findById(id);
    if (!deposit) return res.status(404).json({ message: 'Deposit not found' });

    if (deposit.status === 'approved' || deposit.status === 'rejected') {
      return res.status(400).json({ message: `Deposit already ${deposit.status}` });
    }

    deposit.status = status;
    await deposit.save();

    if (status === 'approved') {
      const user = await User.findById(deposit.userId);
      if (!user) return res.status(404).json({ message: 'User not found' });

      user.balance += deposit.amount;
      await user.save();
    }

    res.json({ message: `Deposit ${status}`, deposit });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// --- ADMIN: DASHBOARD STATS ---
exports.getDashboardStats = async (req, res) => {
  try {
    const today = new Date();
    today.setHours(0, 0, 0, 0);
    const newUsers = await User.countDocuments({ createdAt: { $gte: today } });

    const [depositTotal, withdrawalTotal, deposits, withdrawals] = await Promise.all([
      Deposit.aggregate([{ $group: { _id: null, total: { $sum: "$amount" } } }]),
      Withdrawal.aggregate([{ $group: { _id: null, total: { $sum: "$amount" } } }]),
      Deposit.find(),
      Withdrawal.find(),
    ]);

    const transactions = [
      ...deposits.map(tx => ({
        _id: tx._id,
        type: 'deposit',
        amount: tx.amount,
        status: tx.status,
        createdAt: tx.createdAt,
      })),
      ...withdrawals.map(tx => ({
        _id: tx._id,
        type: 'withdrawal',
        amount: tx.amount,
        status: tx.status,
        createdAt: tx.createdAt,
      })),
    ].sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));

    res.json({
      stats: {
        newUsers,
        totalDeposit: depositTotal[0]?.total || 0,
        totalWithdrawal: withdrawalTotal[0]?.total || 0,
      },
      transactions,
    });
  } catch (error) {
    console.error("Dashboard Stats Error:", error);
    res.status(500).json({ message: "Server error" });
  }
};