const CardRequest = require('../models/Card');
const User = require('../models/Users');
const sendNotification = require('../utils/sendNotification');

const CARD_COST = 2499;

exports.requestCard = async (req, res) => {
  const { cardType } = req.body;

  if (!['Debit', 'Credit'].includes(cardType)) {
    return res.status(400).json({ message: 'Invalid card type' });
  }

  try {
    const user = await User.findById(req.user.id);
    if (!user) return res.status(404).json({ message: 'User not found' });

    if (user.balance < CARD_COST) {
      return res.status(400).json({ message: 'Insufficient balance to request card' });
    }

    // Deduct balance
    user.balance -= CARD_COST;
    await user.save();

    // Calculate expiry date (1 year from now)
    const expiryDate = new Date();
    expiryDate.setFullYear(expiryDate.getFullYear() + 1);

    // Save card request with expiryDate
    const cardRequest = new CardRequest({
      userId: user._id,
      cardType,
      expiryDate,
    });

    await cardRequest.save();
    await sendNotification({
      userId: user._id,
      title: "Card Requested Successful",
      message: `Your card will be reviewed and approved by this admins.`,
      icon: "arrow-down", // Choose icon based on context (optional)
    });
    res.status(201).json({ message: 'Card request submitted', cardRequest });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

exports.getMyCards = async (req, res) => {
  try {
    // Get all card requests for the user
    const cards = await CardRequest.find({ userId: req.user.id }).sort({ createdAt: -1 });

    // Prepare response with card details only if approved
    const formatted = await Promise.all(
      cards.map(async (card) => {
        const cardObj = card.toObject();

        if (card.status === "approved") {
          const fullCard = await CardRequest.findById(card._id).select("cardNumber cvv expiryDate");
          return {
            ...cardObj,
            cardNumber: fullCard.cardNumber,
            cvv: fullCard.cvv,
            expiryDate: fullCard.expiryDate,
          };
        }

        return cardObj; // no sensitive fields
      })
    );

    res.json(formatted);
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};
const generateCardDetails = () => {
  const cardNumber = Array(16)
    .fill(0)
    .map(() => Math.floor(Math.random() * 10))
    .join("");

  const cvv = Math.floor(100 + Math.random() * 900).toString();
  const expiryDate = `${String(new Date().getMonth() + 1).padStart(2, '0')}/${(new Date().getFullYear() + 3)
    .toString()
    .slice(-2)}`;

  return { cardNumber, cvv, expiryDate };
};

exports.approveCard = async (req, res) => {
  const { id } = req.params;

  try {
    const card = await CardRequest.findById(id);
    if (!card) return res.status(404).json({ message: 'Card request not found' });

    if (card.status === 'approved') {
      return res.status(400).json({ message: 'Card already approved' });
    }

    const { cardNumber, cvv, expiryDate } = generateCardDetails();

    card.status = 'approved';
    card.cardNumber = cardNumber;
    card.cvv = cvv;
    card.expiryDate = expiryDate;

    await card.save();

    res.json({ message: 'Card approved and details generated', cardId: card._id });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

