// controllers/depositController.js
const Deposit = require('../models/Deposit');
const User = require('../models/Users');
const sendEmail = require('../utils/sendEmail');
const { depositPendingEmail } = require('../utils/emailTemplates');
const sendNotification = require('../utils/sendNotification');

// Step 1: User initiates a deposit
exports.initiateDeposit = async (req, res) => {
  const { amount, depositType } = req.body;

  if (!amount || !depositType) {
    return res.status(400).json({ message: 'Amount and deposit type are required' });
  }

  try {
    const deposit = new Deposit({
      userId: req.user.id,
      amount,
      depositType,
      status: 'awaiting-payment',
    });

    await deposit.save();
await sendNotification({
      userId: req.user.id,
      title: "Deposit Successful",
      message: `Your deposit of $${amount} was successful.`,
      icon: "arrow-down", // Choose icon based on context (optional)
    });
    if (depositType === 'crypto') {
      return res.status(201).json({ success: true, message: 'Crypto deposit initiated, select method', depositType, deposit });
    }
    if (depositType === 'cash') {
      return res.status(201).json({ success: true, message: 'Cash deposit initiated proceed', depositType, deposit });
    }
    if (depositType === 'paypal') {
      return res.status(201).json({ success: true, message: 'Paypal deposit initiated proceed ', depositType, deposit });
    }
    // If PayPal or cash, notify admin
    // await sendEmail(
    //   process.env.ADMIN_EMAIL,
    //   'New Deposit Request (Pending Payment Details)',
    //   `A new ${depositType} deposit request was made for $${amount}. Please provide payment details.`,
    //   depositPendingEmail(depositType, amount)
    // );

    res.status(201).json({ success: true, message: 'Deposit request initiated', deposit });
  } catch (err) {
    console.error(err);
    res.status(500).json({ message: 'Server error' });
  }
};

// Step 2: User submits payment proof
exports.submitDeposit = async (req, res) => {
  const { id } = req.params;

  try {
    const deposit = await Deposit.findById(id);
    if (!deposit) return res.status(404).json({ message: 'Deposit not found' });
    if (deposit.userId.toString() !== req.user.id) {
      return res.status(403).json({ message: 'Unauthorized' });
    }

    deposit.status = 'submitted';
    await deposit.save();

    // Notify admin to approve
    // await sendEmail(
    //   process.env.ADMIN_EMAIL,
    //   'Deposit Awaiting Approval',
    //   `A user has submitted payment for a $${deposit.amount} ${deposit.depositType} deposit.`,
    //   `<p>Please log in to admin dashboard to approve or decline this deposit.</p>`
    // );

    res.json({ message: 'Payment submitted and pending approval', deposit });
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};

exports.getMyDeposits = async (req, res) => {
  try {
    const deposits = await Deposit.find({ userId: req.user.id }).sort({ createdAt: -1 });
    res.json(deposits);
  } catch (err) {
    res.status(500).json({ message: 'Server error' });
  }
};


// controllers/adminDepositController.js
exports.getAllDeposits = async (req, res) => {
  const deposits = await Deposit.find().populate('userId', 'name email').sort({ createdAt: -1 });
  res.json(deposits);
};

exports.attachPaymentDetails = async (req, res) => {
  const { id } = req.params;
  const { paymentDetails } = req.body;

  const deposit = await Deposit.findById(id).populate('userId');
  if (!deposit) return res.status(404).json({ message: 'Deposit not found' });

  deposit.paymentDetails = paymentDetails;
  await deposit.save();

  await sendEmail(
    deposit.userId.email,
    'Payment Instructions for Your Deposit',
    `Your deposit is awaiting payment. Please use the info below:`,
    `<p>${paymentDetails}</p>`
  );

  res.json({ message: 'Payment details attached and sent to user' });
};

exports.approveDeposit = async (req, res) => {
  const { id } = req.params;
  const deposit = await Deposit.findById(id).populate('userId');
  if (!deposit) return res.status(404).json({ message: 'Deposit not found' });

  deposit.status = 'approved';
  await deposit.save();

  // Credit user balance
  const user = await User.findById(deposit.userId._id);
  user.balance += deposit.amount;
  await user.save();

  await sendEmail(
    user.email,
    'Deposit Approved - Credit Alert',
    `Your $${deposit.amount} deposit has been approved and your account credited.`,
    `<p>Thanks for banking with us.</p>`
  );

  res.json({ message: 'Deposit approved and user credited' });
};

exports.declineDeposit = async (req, res) => {
  const { id } = req.params;
  const deposit = await Deposit.findById(id).populate('userId');
  if (!deposit) return res.status(404).json({ message: 'Deposit not found' });

  deposit.status = 'declined';
  await deposit.save();

  await sendEmail(
    deposit.userId.email,
    'Deposit Declined',
    `Your $${deposit.amount} deposit was declined.`,
    `<p>Please contact support if you believe this is an error.</p>`
  );

  res.json({ message: 'Deposit declined and user notified' });
};
