const Saving = require('../models/Saving');
const User = require('../models/Users');

// Create new saving
exports.createSaving = async (req, res) => {
  const { title, amount, endDate, category } = req.body;
  const userId = req.user.id;

  if (!title || !amount || !endDate) {
    return res.status(400).json({ message: 'All fields are required' });
  }

  const days = Math.ceil((new Date(endDate) - new Date()) / (1000 * 60 * 60 * 24));
  if (days <= 0) {
    return res.status(400).json({ message: 'End date must be in the future' });
  }

  const dailyAmount = parseFloat((amount / days).toFixed(2));

  const saving = new Saving({
    userId,
    title,
    amount,
    endDate,
    category,
    dailyAmount,
    balance: 0,
  });

  await saving.save();

  res.status(201).json({ message: 'Saving plan created', saving });
};

// Auto-debit logic (to be called by a cron job)
exports.processDailySavings = async () => {
  const today = new Date();
  const activeSavings = await Saving.find({
    completed: false,
    endDate: { $gte: today },
  });

  for (let saving of activeSavings) {
    const user = await User.findById(saving.userId);

    if (user.balance >= saving.dailyAmount) {
      user.balance -= saving.dailyAmount;
      saving.balance += saving.dailyAmount;

      // If today is end date, mark as complete and credit back
      if (new Date(saving.endDate).toDateString() === today.toDateString()) {
        user.balance += saving.balance;
        saving.completed = true;
      }

      await user.save();
      await saving.save();
    }
    // else: You could notify user of insufficient funds
  }
};
exports.getUserSavings = async (req, res) => {
  try {
    const userId = req.user.id;

    const savings = await Saving.find({ userId }).sort({ createdAt: -1 });

    res.status(200).json({ success: true, savings });
  } catch (err) {
    console.error('Error fetching user savings:', err);
    res.status(500).json({ success: false, message: 'Server error' });
  }
};